(function ($) {
  'use strict';

  /**
   * Base class for view components
   * @class
   * @memberof psol.components
   * @extends psol.components.Base
   * @param {String} componentName The internal name of the view component
   */
  psol.components.ViewBase = function (componentName) {
    psol.components.Base.call(this, componentName);
    this._props.container = undefined;
    this._props.jsrenderHelpers = {};
  };

  // inheritance
  psol.components.ViewBase.prototype = Object.create(psol.components.Base.prototype);
  psol.components.ViewBase.constructor = psol.components.ViewBase;

  /**
   * Render the template
   * @protected
   * @param   {String} templateName The name of the template in the form 'tmpl-INTERNAL_COMPONENT_NAME'
   * @param   {Object} data         The data for the template
   * @returns {String} The rendered HTML DOM
   */
  psol.components.ViewBase.prototype._render = function (templateName, data) {
    var tmpl;
    if ($.templates.hasOwnProperty(templateName)) {
      tmpl = $.templates[templateName];
    }

    if (!tmpl) {
      tmpl = $.templates(templateName, '#' + templateName);
    }
    return tmpl.render(data, this._props.jsrenderHelpers);
  };


  /**
   * Load the template for this view component
   * @protected
   * @returns {jQuery.Deferred} A {@link http://api.jquery.com/category/deferred-object/ jQuery.Deferred} object, which resolves when the view is rendered
   */
  psol.components.ViewBase.prototype._loadTemplate = function () {
    var def = $.Deferred();
    var templateName = this._props.templateName || this._props.componentName;
    if ($('#tmpl-' + templateName).length === 0) {
      $.ajax({
        url: psol.components.componentsPath + 'tmpl/' + templateName + '.tmpl'
      }).done(function (data) {
        $('body').append(data);
        def.resolve();
      });
    } else {
      def.resolve();
    }

    return def.promise();
  };


  // brite view life cycle
  /**
   * brite.js callback which is called just after brite.display and must return the DOM
   * @protected
   * @param   {Object} data   The data for the template
   * @param   {Object} config The brite config object
   * @returns {String} The rendered HTML DOM
   */
  /*eslint-disable no-unused-vars*/
  psol.components.ViewBase.prototype._viewCreate = function (data, config) {
    var templateName = this._props.templateName || this._props.componentName;
    return this._render('tmpl-' + templateName, data);
  };
  /*eslint-enable no-unused-vars*/


  /**
   * brite.js callback which is called after [ViewBase._viewCreate]{@link psol.components.ViewBase#_viewCreate} but before the content is displayed
   * ViewBase class implmentation does nothing
   * @protected
   * @param   {Object} data   The data for the template
   * @param   {Object} config The brite config object
   */
  /*eslint-disable no-unused-vars*/
  psol.components.ViewBase.prototype._viewInit = function (data, config) {
    /* ViewBase class does noting here */
    return;
  };
  /*eslint-enable no-unused-vars*/

  /**
   * brite.js callback which is called after [ViewBase._viewInit]{@link psol.components.ViewBase#_viewInit}
   * ViewBase class implmentation does nothing
   * @protected
   * @param   {Object} data   The data for the template
   * @param   {Object} config The brite config object
   */
  /*eslint-disable no-unused-vars*/
  psol.components.ViewBase.prototype._viewPostDisplay = function (data, config) {
    /* ViewBase class does noting here */
    return;
  };
  /*eslint-enable no-unused-vars*/


  /**
   * brite.js callback which is called when the view get destroyed
   * ViewBase class implmentation does nothing
   * @protected
   */
  psol.components.ViewBase.prototype._viewDestroy = function () {
    /* ViewBase class does noting here */
    return;
  };


  /**
   * brite.js View events
   * @protected
   * @returns {Object} The view events
   */
  psol.components.ViewBase.prototype._viewEvents = function () {
    return {};
  };


  /* public API methods */


  /**
   * Initialize the component
   * @param   {Object} settings The settings for this component
   */
  psol.components.ViewBase.prototype.init = function (settings) {
    psol.components.Base.prototype.init.call(this, settings);


    var that = this;

    // register brite view
    brite.registerView(this._props.componentName, {
      loadTmpl: false,
      loadCss: false
    }, {
      create: function (data, config) {
        return that._viewCreate(data, config);
      },
      init: function (data, config) {
        return that._viewInit(data, config);
      },
      postDisplay: function (data, config) {
        return that._viewPostDisplay(data, config);
      },
      destroy: function () {
        return that._viewDestroy();
      },
      events: that._viewEvents()
    });
  };

  /**
   * Displays the component
   * @returns {jQuery.Deferred} A {@link http://api.jquery.com/category/deferred-object/ jQuery.Deferred} object, which resolves when the view is rendered
   */
  psol.components.ViewBase.prototype.show = function () {
    var that = this;
    var templateName = this._props.templateName || this._props.componentName;
    var def = $.Deferred();
    that._loadTemplate().done(function () {
      brite.display(templateName, that._props.container).done(function () {
        def.resolve();
      });
    });

    return def.promise();
  };


  /**
   * Call this when resize the container, ViewBase implementation does nothing
   */
  psol.components.ViewBase.prototype.resizeContainer = function () {
    return;
  };


}(jQuery));
