# URI.js\n\nURI.js is an [RFC 3986](http://www.ietf.org/rfc/rfc3986.txt) compliant, scheme extendable URI parsing/validating/resolving library for all JavaScript environments (browsers, Node.js, etc).\nIt is also compliant with the IRI ([RFC 3987](http://www.ietf.org/rfc/rfc3987.txt)), IDNA ([RFC 5890](http://www.ietf.org/rfc/rfc5890.txt)), IPv6 Address ([RFC 5952](http://www.ietf.org/rfc/rfc5952.txt)), IPv6 Zone Identifier ([RFC 6874](http://www.ietf.org/rfc/rfc6874.txt)) specifications.\n\nURI.js has an extensive test suite, and works in all (Node.js, web) environments. It weighs in at 6.2kb (gzipped, 16kb deflated).\n\n## API\n\n### Parsing\n\n\tURI.parse(\"uri://user:pass@example.com:123/one/two.three?q1=a1&q2=a2#body\");\n\t//returns:\n\t//{\n\t//  scheme : \"uri\",\n\t//  userinfo : \"user:pass\",\n\t//  host : \"example.com\",\n\t//  port : 123,\n\t//  path : \"/one/two.three\",\n\t//  query : \"q1=a1&q2=a2\",\n\t//  fragment : \"body\"\n\t//}\n\n### Serializing\n\n\tURI.serialize({scheme : \"http\", host : \"example.com\", fragment : \"footer\"}) === \"http://example.com/#footer\"\n\n### Resolving\n\n\tURI.resolve(\"uri://a/b/c/d?q\", \"../../g\") === \"uri://a/g\"\n\n### Normalizing\n\n\tURI.normalize(\"HTTP://ABC.com:80/%7Esmith/home.html\") === \"http://abc.com/~smith/home.html\"\n\n### Comparison\n\n\tURI.equal(\"example://a/b/c/%7Bfoo%7D\", \"eXAMPLE://a/./b/../b/%63/%7bfoo%7d\") === true\n\n### IP Support\n\n\t//IPv4 normalization\n\tURI.normalize(\"//192.068.001.000\") === \"//192.68.1.0\"\n\n\t//IPv6 normalization\n\tURI.normalize(\"//[2001:0:0DB8::0:0001]\") === \"//[2001:0:db8::1]\"\n\n\t//IPv6 zone identifier support\n\tURI.parse(\"//[2001:db8::7%25en1]\");\n\t//returns:\n\t//{\n\t//  host : \"2001:db8::7%en1\"\n\t//}\n\n### IRI Support\n\n\t//convert IRI to URI\n\tURI.serialize(URI.parse(\"http://examplé.org/rosé\")) === \"http://xn--exampl-gva.org/ros%C3%A9\"\n\t//convert URI to IRI\n\tURI.serialize(URI.parse(\"http://xn--exampl-gva.org/ros%C3%A9\"), {iri:true}) === \"http://examplé.org/rosé\"\n\n### Options\n\nAll of the above functions can accept an additional options argument that is an object that can contain one or more of the following properties:\n\n*\t`scheme` (string)\n\n\tIndicates the scheme that the URI should be treated as, overriding the URI's normal scheme parsing behavior.\n\n*\t`reference` (string)\n\n\tIf set to `\"suffix\"`, it indicates that the URI is in the suffix format, and the validator will use the option's `scheme` property to determine the URI's scheme.\n\n*\t`tolerant` (boolean, false)\n\n\tIf set to `true`, the parser will relax URI resolving rules.\n\n*\t`absolutePath` (boolean, false)\n\n\tIf set to `true`, the serializer will not resolve a relative `path` component.\n\n*\t`iri` (boolean, false)\n\n\tIf set to `true`, the serializer will unescape non-ASCII characters as per [RFC 3987](http://www.ietf.org/rfc/rfc3987.txt).\n\n*\t`unicodeSupport` (boolean, false)\n\n\tIf set to `true`, the parser will unescape non-ASCII characters in the parsed output as per [RFC 3987](http://www.ietf.org/rfc/rfc3987.txt).\n\n*\t`domainHost` (boolean, false)\n\n\tIf set to `true`, the library will treat the `host` component as a domain name, and convert IDNs (International Domain Names) as per [RFC 5891](http://www.ietf.org/rfc/rfc5891.txt).\n\n## Scheme Extendable\n\nURI.js supports inserting custom [scheme](http://en.wikipedia.org/wiki/URI_scheme) dependent processing rules. Currently, URI.js has built in support for the following schemes:\n\n*\thttp \\[[RFC 2616](http://www.ietf.org/rfc/rfc2616.txt)\\]\n*\thttps \\[[RFC 2818](http://www.ietf.org/rfc/rfc2818.txt)\\]\n*\tmailto \\[[RFC 6068](http://www.ietf.org/rfc/rfc6068.txt)\\]\n*\turn \\[[RFC 2141](http://www.ietf.org/rfc/rfc2141.txt)\\]\n*\turn:uuid \\[[RFC 4122](http://www.ietf.org/rfc/rfc4122.txt)\\]\n\n### HTTP Support\n\n\tURI.equal(\"HTTP://ABC.COM:80\", \"http://abc.com/\") === true\n\n### Mailto Support\n\n\tURI.parse(\"mailto:alpha@example.com,bravo@example.com?subject=SUBSCRIBE&body=Sign%20me%20up!\");\n\t//returns:\n\t//{\n\t//\tscheme : \"mailto\",\n\t//\tto : [\"alpha@example.com\", \"bravo@example.com\"],\n\t//\tsubject : \"SUBSCRIBE\",\n\t//\tbody : \"Sign me up!\"\n\t//}\n\n\tURI.serialize({\n\t\tscheme : \"mailto\",\n\t\tto : [\"alpha@example.com\"],\n\t\tsubject : \"REMOVE\",\n\t\tbody : \"Please remove me\",\n\t\theaders : {\n\t\t\tcc : \"charlie@example.com\"\n\t\t}\n\t}) === \"mailto:alpha@example.com?cc=charlie@example.com&subject=REMOVE&body=Please%20remove%20me\"\n\n### URN Support\n\n\tURI.parse(\"urn:example:foo\");\n\t//returns:\n\t//{\n\t//\tscheme : \"urn\",\n\t//\tnid : \"example\",\n\t//\tnss : \"foo\",\n\t//}\n\n#### URN UUID Support\n\n\tURI.parse(\"urn:uuid:f81d4fae-7dec-11d0-a765-00a0c91e6bf6\");\n\t//returns:\n\t//{\n\t//\tscheme : \"urn\",\n\t//\tnid : \"example\",\n\t//\tuuid : \"f81d4fae-7dec-11d0-a765-00a0c91e6bf6\",\n\t//}\n\n## Usage\n\nTo load in a browser, use the following tag:\n\n\t<script type=\"text/javascript\" src=\"uri-js/dist/es5/uri.all.min.js\"></script>\n\nTo load in a CommonJS (Node.js) environment, first install with npm by running on the command line:\n\n\tnpm install uri-js\n\nThen, in your code, load it using:\n\n\tconst URI = require(\"uri-js\");\n\nIf you are writing your code in ES6+ (ESNEXT) or TypeScript, you would load it using:\n\n\timport * as URI from \"uri-js\";\n\nOr you can load just what you need using named exports:\n\n\timport { parse, serialize, resolve, resolveComponents, normalize, equal, removeDotSegments, pctEncChar, pctDecChars, escapeComponent, unescapeComponent } from \"uri-js\";\n\n## Breaking changes\n\n### Breaking changes from 3.x\n\nURN parsing has been completely changed to better align with the specification. Scheme is now always `urn`, but has two new properties: `nid` which contains the Namspace Identifier, and `nss` which contains the Namespace Specific String. The `nss` property will be removed by higher order scheme handlers, such as the UUID URN scheme handler.\n\nThe UUID of a URN can now be found in the `uuid` property.\n\n### Breaking changes from 2.x\n\nURI validation has been removed as it was slow, exposed a vulnerabilty, and was generally not useful.\n\n### Breaking changes from 1.x\n\nThe `errors` array on parsed components is now an `error` string.\n\n## License ([Simplified BSD](http://en.wikipedia.org/wiki/BSD_licenses#2-clause))\n\nCopyright 2011 Gary Court. All rights reserved.\n\nRedistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:\n\n1.\tRedistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.\n\n2.\tRedistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.\n\nTHIS SOFTWARE IS PROVIDED BY GARY COURT \"AS IS\" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL GARY COURT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.\n\nThe views and conclusions contained in the software and documentation are those of the authors and should not be interpreted as representing official policies, either expressed or implied, of Gary Court.\n